(* ::Package:: *)

(* ::Title:: *)
(*Analysis Script for LD sensor and Cantilever*)


(* ::Text:: *)
(*History: *)
(*14.03.2023 Testing phase of the script*)


BeginPackage["LVactuatorAnalysis`"];
EvaluateMeasurement::usage="EavaluateMeasurement[filename], takes a path to a measurment file. The output is a list of rules containing metadata, evaluted quantities, and rawdata.";
MinFilterManual::usage=" "; 
MaxFilterManual::usage=" "; 
AvgManual::usage="AvgManual[data,r] takes a dataset data and splits it into parts of length r. An average is computed on each part. This allows size reduction of large datasetst";
BendingAmplidute::usage = "BendingAmplidute[data] takes temporal data of the laserdisplacement sensor and computes the actuators amplitude";
AvergePower::usage = "AvergePower[data] takes temporal data of the power and computes an average value.";
RiseAndFallTimes::usage = "RiseAndFallTimes[voltagedata, amplitudedata, {voltage, frequency,\"rect_bp\",\"Bipolar\"}]\nRiseAndFallTimes[voltagedata,amplitudedata,{voltage,frequency,\"rect_up\",\"Unipolar\"}]";
DataSelect::usage = "DataSelect[EvaluateMeasurement[filename]] is a function to select specific data. \nDataSelect[result_,{xvar_String,yvar_String},option_String->value_]\nDataSelect[result_,{xvar_String,yvar_String},{option1_String->value1_,option2_String->value2_}] ";
GenerateDataAndPlots::usage = "Dont use GenerateDataAndPlots[folder_], its not updated. Use GenerateDataAndPlots[folder,result]";
GetFrequency::usage = "GetFrequency[filename] returns the frequency value of the used voltage signal as a physical quantity.";
GetVoltage::usage = "GetVoltage[filename] returns the voltage amplitude value of the used voltage signal as a physical quantity.";
GetWaveform::usage = "GetWaveform[filename] returns the name of the waveform of the used voltage signal as a String.";
GetRepetition::usage = "Integer value of the number of measurement with the same condition";
SetToZero::usage = "Apply to eg Cuurent data to compensate for potential offset";
DeltaAngle::usage = "Calculates a difference of the bending angle from a given Amplitude and distance value of the laser spot";
DistanceDataFromFileName::usage = "Returns the correct distance value from a measurmentinfo data and filename";
ImportInfo::usage = "Imports the Info File that is saved in the parent directory";
a::usage = "position of laser displacement sensor in mm";
l::usage = "length of cantilever in mm";
resfreq::usage = "Average of cantilever resonance frequency in Hz";
EI::usage = "Modulua times moment of Inertia in N m\.b2";
Deflection::usage = "Cantilever deflection as dunction of cantilever dimensions and input force"


(* ::Section:: *)
(*Evaluate Measurement*)


EvaluateMeasurement[filename_]:=Module[{header,repetition,data,force,voltage=0,voltagedata,currentdata,powerdata,displacementdata, metadata,frequency=0, amplitude, waveform=0, polarity,bendamp=1,avgpower,rise,fall,result,error,F},
data=Import[filename];
header=Flatten[data[[1]]];


(*Print[dis]*);
frequency=Quantity[GetFrequency[filename],"Hertz"];
voltage=Quantity[1000*GetVoltage[filename],"Volts"];
waveform=ToString[GetWaveform[filename]];
repetition=GetRepetition[filename];
If[Not@NumberQ[GetFrequency[filename]] && Not@NumberQ[GetVoltage[filename]] && Not@StringQ[waveform],Print["Error::EvaluateMeasurement::01 -> No data taken from filename"]];
If[Not@NumberQ[GetFrequency[filename]],Print["Error::EvaluateMeasurement::01a -> No frequency data taken from filename"]];
If[Not@NumberQ[GetVoltage[filename]],Print["Error::EvaluateMeasurement::01b -> No voltage data taken from filename"]];
If[Not@StringQ[waveform],Print["Error::EvaluateMeasurement::01c -> No waveform data taken from filename"]];
polarity=Switch[waveform,"sinus_bp","Bipolar","rect_bp","Bipolar","rect_up","Unipolar"];
If[Not@StringQ[polarity],Print["Error::EvaluateMeasurement::02 -> No polarity information available"]];
(*Metadata*) 
metadata={"Frequency"->frequency,"Voltage"->voltage,"Waveform"->waveform,"Polarity"->polarity,"Repetition"->repetition};
(*timeseries data and filtering*)
data=data[[2;;-1]];
voltagedata={#[[1]],#[[3]]}&/@data;
If[Length[voltagedata]==0,Print["Error::EvaluateMeasurement::03a -> No data from the voltage monitor"]];
currentdata=SetToZero[{#[[1]],#[[2]]}&/@data];
If[Length[currentdata]==0,Print["Error::EvaluateMeasurement::03b -> No data from the current monitor"]];
(*currentdata=BandstopFilter[TimeSeries[currentdata],{Quantity[48,"Hertz"],Quantity[52,"Hertz"]}]["Path"];*);
powerdata=Table[{currentdata[[i,1]],currentdata[[i,2]]*voltagedata[[i,2]]},{i,1,Length[currentdata]}];
(*powerdata={#[[1]],#[[2]] #[[3]]}&/@data*);
displacementdata={#[[1]],#[[4]]}&/@data;
If[Length[displacementdata]==0,Print["Error::EvaluateMeasurement::03c -> No data from the laser sensor"]];
(*displacementdata=BandstopFilter[TimeSeries[displacementdata],{Quantity[48,"Hertz"],Quantity[52,"Hertz"]}]["Path"];*);
(*amplitude*)
bendamp=Quantity[BendingAmplidute[voltagedata,displacementdata],"Millimeters"];
If[Not@QuantityQ[bendamp],Print["Error::EvaluateMeasurement::04a -> No amplitude computed"]];
result=Append[metadata,"BendingAmplitude"->bendamp];
(*blocked force*)
force=F/.Solve[-bendamp==Deflection[a,l-Quantity[1,"Millimeters"],l,F],F][[1]];
result=Append[result,"Force"->force];
If[Not@QuantityQ[force],Print["Error::EvaluateMeasurement::04b -> No blocked force computed"]];
(*average power*)
avgpower=AvergePower[powerdata];
If[Not@QuantityQ[avgpower],Print["Error::EvaluateMeasurement::04c -> No avg power computed"]];
result=Append[result,"AvgPowerConsumption"->avgpower];

(*add rawdata to output*)
result=Append[result,"Header"->header];
result=Append[result,"Rawdata"->data];
result
]


(* ::Input:: *)
(**)


(* ::Section:: *)
(*Cantilever Measures and Functions*)


a=Quantity[30.12,"Millimeters"];
l=Quantity[67,"Millimeters"];
resfreq=55.576;
EI=UnitConvert[(4 \[Pi]^2 Quantity[resfreq,"Hertz"]^2)/3.5^2 Quantity[93.6/100,"Milligrams"/"Millimeters"]  Quantity[67,"Millimeters"]^4,"Newtons" "Meters"^2];


Deflection[x_,a_,l_,F_]:=Piecewise[{{(F x^2)/(6 EI) (3a-x),Quantity[0,"Millimeters"]<=x<a},{(F a^2)/(6 EI) (3x-a),a<=x<=l}}]


(* ::Section:: *)
(*Helping Functions*)


(* ::Subsection:: *)
(*Filters for Min Max detection*)


MinFilterManual[data_,r_Integer]:=Table[First[MinimalBy[data[[i;;i+r]],Last]],{i,1,Length[data]-r}]
MaxFilterManual[data_,r_Integer]:=Table[First[MaximalBy[data[[i;;i+r]],Last]],{i,1,Length[data]-r}]


AvgManual[data_,r_Integer]:=Table[Mean[data[[(i-1)r+1;;i r+1]]],{i,1,Floor[Length[data]/r]-1}]


(* ::Input:: *)
(**)


(* ::Subsection:: *)
(*Data Selecting Tools*)


DataSelect[result_,{xvar_String,yvar_String},option_String->value_]:=Sort[{xvar,yvar}/.#&/@Select[result,(option/.#)==value&]]
DataSelect[result_,{xvar_String,yvar_String},{option1_String->value1_,option2_String->value2_}]:=Sort[{xvar,yvar}/.#&/@Select[result,(option1/.#)==value1&&(option2/.#)==value2&]]
DataSelect[result_,{xvar_String,yvar_String},{option1_String->value1_,option2_String->value2_,option3_String->value3_}]:=Sort[{xvar,yvar}/.#&/@Select[result,(option1/.#)==value1&&(option2/.#)==value2&&(option3/.#)==value3&]]


(* ::Subsection:: *)
(*MetaData (GetValue)*)


GetFrequency[filename_]:=ToExpression[StringTake[filename,{#[[1,2]]+1,#[[2,1]]-1}&@StringPosition[filename,{"F_","Hz_"}]]]
GetVoltage[filename_]:=ToExpression[StringTake[filename,{#[[1,2]]+1,#[[2,1]]-1}&@StringPosition[filename,{"U_","V_"}]]]
GetWaveform[filename_]:=ToExpression[StringTake[filename,{#[[1,2]]+1,#[[2,1]]-1}&@StringPosition[filename,{"W_","_F"}]]]
GetRepetition[filename_]:=ToExpression[StringTake[FileBaseName@filename,-1]]


(* ::Subsection:: *)
(*Compensate Current Bias*)


SetToZero[data_]:=Module[{experimentinterval=5, reference},
reference=Select[data,Between[#[[1]],{experimentinterval,data[[-1,1]]}]&];
reference=Mean[reference][[2]];
{#[[1]],#[[2]]-reference}&/@data
]


(* ::Subsection:: *)
(*Angle Analysis Not Needed*)


DeltaAngle[amplitude_,distance_]:=Tanh[amplitude/distance]/Degree


DistanceDataFromFileName[filename_,MeasurementInfo_]:=Module[{save1,save2,result},
result=Flatten[Select[MeasurementInfo,(save1=StringTake[First[#],{-3,-1}])==(save2=StringTake[filename,{Flatten[StringPosition[filename,{"_W"}]][[1]]-3,Flatten[StringPosition[filename,{"_W"}]][[1]]-1}])&]][[3]];
(*Print[save1];Print[save2]*);
result]


ImportInfo[]:=Module[{cd=Directory[],pd=ParentDirectory[Directory[]],info},
SetDirectory[pd];
info=FileNames["*MeasurementInfo.csv"][[1]];
info=Import[info];
SetDirectory[cd];
info
]


(* ::Section:: *)
(*Analysis Functions*)


(* ::Subsection:: *)
(*Bending Amplitude*)


BendingAmplidute[voltagedata_,amplitudedata_]:=Module[{selection,len=2.5,start,end,zero,max},
selection=Select[voltagedata,Abs[#[[2]]]>=100&];
start=selection[[1,1]];
end=selection[[-1,1]];

max=Mean[Select[amplitudedata,Between[#[[1]],{start+len,end}]&]][[2]];
If[Not@NumberQ[max],Print["Error::Cantilever Bending Amplitude -> deflection data error"]];

zero=Mean[Select[amplitudedata,Between[#[[1]],{amplitudedata[[1,1]],start}]&]][[2]];
If[Not@NumberQ[zero],Print["Error::Cantilever Bending Amplitude -> zero level error"]];

max-zero

]


(* ::Subsection:: *)
(*Average Power Consumption*)


AvergePower[powerdata_]:=Module[{ts,path,int,experimentinterval=5},
path=powerdata;
path=Select[path,Between[#[[1]],{1,1+experimentinterval}]&&#[[2]]>0&];
If[Length[path]<2,Print["Error::AvaragePower[] -> no data"]];
int=Interpolation[Select[path,#[[2]]>0&],InterpolationOrder->1];
Quantity[Integrate[int[x],{x,path[[1,1]],path[[-1,1]]}]/experimentinterval,"Milliwatts"]
]


(* ::Section:: *)
(*Generate Data Plots and Export*)


(* ::Subsection:: *)
(*PlotOptions*)


PlotOptions={
Frame->{True,True,False,False},
FrameStyle->Directive[Black,Thickness[0.003]],
PlotRange->{{0,Automatic},All},
Joined->True,
PlotMarkers->Automatic,
ImageSize->300,
AspectRatio->0.8,
FrameTicksStyle->{Directive[Black, 12],Directive[Black, 12]},
PlotStyle->{Blue,Red, Black, {Black,Dashed}}
}


(* ::Subsection::Closed:: *)
(*Generate Data and Plots (Fully automated)*)


(* ::Text:: *)
(*Fully Automated Function that also calls Evaluate measurement. NEEDS UPDATE. DONT USE. *)


GenerateDataAndPlots[folder_,voltages_List,frequencies_List]:=Module[{currentdir,files, result,SinBP,RecBP,UniP,UniN,PSinBP,PRecBP,PUniP,PUniN,PAF,PPF,timetable,volt1,volt2,freq1},
(*Set directory to specified folder*)
currentdir=Directory[];
SetDirectory[folder];

(*use the voltages and frequencies used for the measurement series*)
{volt1,volt2}=voltages;
{freq1}=frequencies;

(*import files and evaluate measurements*);
files=FileNames["*.csv"];
result=EvaluateMeasurement/@files;

(*Data for bending amplidute versus freqeuency*)
SinBP={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","BendingAmplitude"},{"Waveform"->"sinus_bp","Voltage"->Quantity[volt1,"Volts"]}];
RecBP={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","BendingAmplitude"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"]}];
UniP={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Frequency","BendingAmplitude"},{"Waveform"->"rect_up","Voltage"->Quantity[volt1,"Volts"]}];
UniN={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Frequency","BendingAmplitude"},{"Waveform"->"rect_up","Voltage"->Quantity[-volt1,"Volts"]}];
PAF=ListPlot[{SinBP,RecBP,UniP,UniN},Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Bending Amplitude (mm)",Black, 12]},PlotLegends->Placed[{Style["Sin BP",10],Style["Rect BP",10], Style["Rect UP+",10],Style["Rect UP-",10]},{Left,Bottom}]];
Export["Plot-BenAmpl-vs-BenFreq-"<>ToString[volt1]<>"V.pdf",PAF];
Export["Data-BenAmpl-vs-BenFreq-"<>ToString[volt1]<>"V.dat",{SinBP,RecBP,UniP,UniN}];

(*Data for power consumption versus freqeuency*)
PSinBP={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"sinus_bp","Voltage"->Quantity[volt1,"Volts"]}];
PRecBP={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"]}];
PUniP={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_up","Voltage"->Quantity[volt1,"Volts"]}];
PUniN={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_up","Voltage"->Quantity[-volt1,"Volts"]}];
PPF=ListPlot[{PSinBP,PRecBP,PUniP,PUniN},Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Power Consumption (mW)",Black, 12]},PlotLegends->Placed[{Style["Sin BP",10],Style["Rect BP",10], Style["Rect UP+",10],Style["Rect UP-",10]},{Left,Top}]];
Export["Plot-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V.pdf",PPF];
Export["Data-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V.dat",{PSinBP,PRecBP,PUniP,PUniN}];

(*rise and falltime filtering*)
timetable={{"Polarity","Rise-Time","Fall-Time"},
Flatten[{"Bipolar",Mean@DataSelect[result,{"RiseTime","FallTime"},{"Waveform"->"rect","Frequency"->Quantity[freq1,"Hertz"]}]}],
Flatten[{"Positive",Mean@DataSelect[result,{"RiseTime","FallTime"},{"Waveform"->"unipolar","Frequency"->Quantity[2 freq1,"Hertz"],"Voltage"->Quantity[volt1,"Volts"]}]}],
Flatten[{"Negative",Mean@DataSelect[result,{"RiseTime","FallTime"},{"Waveform"->"unipolar","Frequency"->Quantity[freq1,"Hertz"],"Voltage"->Quantity[-volt1,"Volts"]}]}]
};
Export["Data-RiseAndFallTime-"<>ToString[2*freq1]<>"Hz-"<>ToString[volt1]<>"V.dat",timetable];
(*reset directory location to original directory*);
SetDirectory[currentdir];
GraphicsRow[{PAF,PPF,TableForm@timetable},ImageSize->650+300]
]


(* ::Subsection:: *)
(*Generate Data and Plots (Half automated)*)


GenerateDataAndPlots[folder_,result_,voltages_List,frequencies_List]:=Module[{samplenr=FileBaseName[folder],currentdir,files,
RecBP1a,RecBP1b,RecBP1c,
RecBP2a,RecBP2b,RecBP2c,
RecBP3a,RecBP3b,RecBP3c,
PRecBP1a,PRecBP1b,PRecBP1c,
PRecBP2a,PRecBP2b,PRecBP2c,
PRecBP3a,PRecBP3b,PRecBP3c,
VRecBP1a,VRecBP1b,VRecBP1c,
PAF1,PAF2,PAF3,PPF1,PPF2,PPF3,PAV2,PPAV2,
volt1,volt2,volt3,freq1},
(*Set directory to specified folder*)
currentdir=Directory[];
SetDirectory[folder];

(*use the voltages and frequencies used for the measurement series*)
{volt1,volt2,volt3}=voltages;
{freq1}=frequencies;

(*Data for blocked force versus freqeuency*)
RecBP1a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->1}];
If[ListQ[RecBP1a],RecBP1a=RecBP1a,RecBP1a={}];
RecBP1b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->2}];
If[ListQ[RecBP1b],RecBP1b=RecBP1b,RecBP1b={}];
RecBP1c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->3}];
If[ListQ[RecBP1c],RecBP1c=RecBP1c,RecBP1c={}];

RecBP2a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->1}];
If[ListQ[RecBP2a],RecBP2a=RecBP2a,RecBP2a={}];
RecBP2b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->2}];
If[ListQ[RecBP2b],RecBP2b=RecBP2b,RecBP2b={}];
RecBP2c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->3}];
If[ListQ[RecBP2c],RecBP2c=RecBP2c,RecBP2c={}];

RecBP3a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->1}];
If[ListQ[RecBP3a],RecBP3a=RecBP3a,RecBP3a={}];
RecBP3b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->2}];
If[ListQ[RecBP3b],RecBP3b=RecBP3b,RecBP3b={}];
RecBP3c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","Force"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->3}];
If[ListQ[RecBP3c],RecBP3c=RecBP3c,RecBP3c={}];

PAF1=ListPlot[Select[{RecBP1a,RecBP1b,RecBP1c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt1]<>" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Blocked Force(mN)",Black, 12]}];

PAF2=ListPlot[Select[{RecBP2a,RecBP2b,RecBP2c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt2]<>" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Blocked Force(mN)",Black, 12]}];

PAF3=ListPlot[Select[{RecBP3a,RecBP3b,RecBP3c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt3]<>" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Blocked Force(mN)",Black, 12]}];

Export[samplenr<>"_Plot-BlForce-vs-BenFreq-"<>ToString[volt1]<>"V.pdf",PAF1];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt1]<>"V_1.dat",RecBP1a];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt1]<>"V_2.dat",RecBP1b];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt1]<>"V_3.dat",RecBP1c];

Export[samplenr<>"_Plot-BlForce-vs-BenFreq-"<>ToString[volt2]<>"V.pdf",PAF2];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt2]<>"V_1.dat",RecBP2a];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt2]<>"V_2.dat",RecBP2b];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt2]<>"V_3.dat",RecBP2c];

Export[samplenr<>"_Plot-BlForce-vs-BenFreq-"<>ToString[volt3]<>"V.pdf",PAF3];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt3]<>"V_1.dat",RecBP3a];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt3]<>"V_2.dat",RecBP3b];
Export[samplenr<>"_Data-BlForce-vs-BenFreq-"<>ToString[volt3]<>"V_3.dat",RecBP3c];


(*Data for power consumption versus freqeuency*)
PRecBP1a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->1}];
If[ListQ[PRecBP1a],PRecBP1a=PRecBP1a,PRecBP1a={}];
PRecBP1b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->2}];
If[ListQ[PRecBP1b],PRecBP1b=PRecBP1b,PRecBP1b={}];
PRecBP1c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt1,"Volts"],"Repetition"->3}];
If[ListQ[PRecBP1c],PRecBP1c=PRecBP1c,PRecBP1c={}];

PRecBP2a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->1}];
If[ListQ[PRecBP2a],PRecBP2a=PRecBP2a,PRecBP2a={}];
PRecBP2b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->2}];
If[ListQ[PRecBP2b],PRecBP2b=PRecBP2b,PRecBP2b={}];
PRecBP2c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt2,"Volts"],"Repetition"->3}];
If[ListQ[PRecBP2c],PRecBP2c=PRecBP2c,PRecBP2c={}];

PRecBP3a={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->1}];
If[ListQ[PRecBP3a],PRecBP3a=PRecBP3a,PRecBP3a={}];
PRecBP3b={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->2}];
If[ListQ[PRecBP3b],PRecBP3b=PRecBP3b,PRecBP3b={}];
PRecBP3c={#[[1]] 2,#[[2]]}&/@DataSelect[result,{"Frequency","AvgPowerConsumption"},{"Waveform"->"rect_bp","Voltage"->Quantity[volt3,"Volts"],"Repetition"->3}];
If[ListQ[PRecBP3c],PRecBP3c=PRecBP3c,PRecBP3c={}];

PPF1=ListPlot[Select[{PRecBP1a,PRecBP1b,PRecBP1c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt1]" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Power Consumption (mW)",Black, 12]},PlotLegends->Placed[{Style["Run 1",10],Style["Run 2",10], Style["Run 3",10],Style["Rect UP-",10]},{Left,Top}]];

PPF2=ListPlot[Select[{PRecBP2a,PRecBP2b,PRecBP2c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt2]" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Power Consumption (mW)",Black, 12]},PlotLegends->Placed[{Style["Run 1",10],Style["Run 2",10], Style["Run 3",10],Style["Rect UP-",10]},{Left,Top}]];

PPF3=ListPlot[Select[{PRecBP3a,PRecBP3b,PRecBP3c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[volt3]" V",Evaluate@PlotOptions,FrameLabel->{Style["Frequency (Hz)",Black, 12],
Style["Power Consumption (mW)",Black, 12]},PlotLegends->Placed[{Style["Run 1",10],Style["Run 2",10], Style["Run 3",10],Style["Rect UP-",10]},{Left,Top}]];

Export[samplenr<>"_Plot-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V.pdf",PPF1];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V_1.dat",PRecBP1a];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V_2.dat",PRecBP1b];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt1]<>"V_3.dat",PRecBP1c];

Export[samplenr<>"_Plot-AvgPowC-vs-BenFreq-"<>ToString[volt2]<>"V.pdf",PPF2];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt2]<>"V_1.dat",PRecBP2a];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt2]<>"V_2.dat",PRecBP2b];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt2]<>"V_3.dat",PRecBP2c];

Export[samplenr<>"_Plot-AvgPowC-vs-BenFreq-"<>ToString[volt3]<>"V.pdf",PPF3];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt3]<>"V_1.dat",PRecBP3a];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt3]<>"V_2.dat",PRecBP3b];
Export[samplenr<>"_Data-AvgPowC-vs-BenFreq-"<>ToString[volt3]<>"V_3.dat",PRecBP3c];

(*Data for blocked force versus Voltage*)
VRecBP1a={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","Force"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->1}];
If[ListQ[VRecBP1a],VRecBP1a=VRecBP1a,VRecBP1a={}];
VRecBP1b={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","Force"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->2}];
If[ListQ[VRecBP1b],VRecBP1b=VRecBP1b,VRecBP1b={}];
VRecBP1c={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","Force"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->3}];
If[ListQ[VRecBP1c],VRecBP1c=VRecBP1c,VRecBP1c={}];

PAV2=ListPlot[Select[{VRecBP1a,VRecBP1b,VRecBP1c},Length[#]>0&],PlotLabel->samplenr<>", "<>ToString[2*freq1]<>" Hz",Evaluate@PlotOptions,FrameLabel->{Style["Voltage (V)",Black, 12],
Style["Blocked Force(mN)",Black, 12]}(*,PlotLegends->Placed[{Style["Sin BP",10],Style["Rect BP",10], Style["Rect UP+",10],Style["Rect UP-",10]},{Left,Bottom}]*)];

Export[samplenr<>"_Plot-BlForce-vs-Voltage-"<>ToString[freq1]<>"Hz.pdf",PAV2];
Export[samplenr<>"_Data-BlForce-vs-Voltage-"<>ToString[freq1]<>"Hz_1.dat",VRecBP1a];
Export[samplenr<>"_Data-BlForce-vs-Voltage-"<>ToString[freq1]<>"Hz_2.dat",VRecBP1b];
Export[samplenr<>"_Data-BlForce-vs-Voltage-"<>ToString[freq1]<>"Hz_3.dat",VRecBP1c];

(*Data for power consumption amplidute versus Voltage*)
PRecBP1a={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","AvgPowerConsumption"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->1}];
If[ListQ[PRecBP1a],PRecBP1a=PRecBP1a,PRecBP1a={}];
PRecBP1b={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","AvgPowerConsumption"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->2}];
If[ListQ[PRecBP1b],PRecBP1b=PRecBP1b,PRecBP1b={}];
PRecBP1c={#[[1]] ,#[[2]]}&/@DataSelect[result,{"Voltage","AvgPowerConsumption"},{"Waveform"->"rect_bp","Frequency"->Quantity[freq1,"Hertz"],"Repetition"->3}];
If[ListQ[PRecBP1c],PRecBP1c=PRecBP1c,PRecBP1c={}];

PPAV2=ListPlot[Select[{PRecBP1a,PRecBP1b,PRecBP1c},Length[#]>0&],PlotLabel->ToString[2*freq1]<>" Hz",Evaluate@PlotOptions,FrameLabel->{Style["Voltage (V)",Black, 12],
Style["Avg. Power Consumption (mV)",Black, 12]}(*,PlotLegends->Placed[{Style["Sin BP",10],Style["Rect BP",10], Style["Rect UP+",10],Style["Rect UP-",10]},{Left,Bottom}]*)];

Export[samplenr<>"_Plot-AvgPowC-vs-Voltage-"<>ToString[freq1]<>"Hz.pdf",PPAV2];
Export[samplenr<>"_Data-AvgPowC-vs-Voltage-"<>ToString[freq1]<>"Hz_1.dat",PRecBP1a];
Export[samplenr<>"_Data-AvgPowC-vs-Voltage-"<>ToString[freq1]<>"Hz_2.dat",PRecBP1b];
Export[samplenr<>"_Data-AvgPowC-vs-Voltage-"<>ToString[freq1]<>"Hz_3.dat",PRecBP1c];


(*reset directory location to original directory*);
SetDirectory[currentdir];
GraphicsGrid[{
{PAF1,PAF2,PAF3},
{PPF1,PPF2,PPF3},
{PAV2,PPAV2}},ImageSize->1000]
]


(* ::Input:: *)
(**)


(* ::Input:: *)
(**)


(* ::Section:: *)
(*End Package*)


(*End[];*)


EndPackage[]
